'use client';

import { Event } from '@/types/event';
import { format, parseISO, isAfter, isBefore, addDays } from 'date-fns';
import { 
  Calendar, 
  MapPin, 
  Users, 
  Ticket, 
  Clock
} from 'lucide-react';
import Link from 'next/link';
import { motion } from 'framer-motion';

interface EventCardProps {
  event: Event;
  index?: number;
}

/**
 * Format event type for display
 */
function formatEventType(type: string): string {
  return type.split('-').map(word => 
    word.charAt(0).toUpperCase() + word.slice(1)
  ).join(' ');
}

/**
 * EventCard component - White card with red border accent
 */
export default function EventCard({ event, index = 0 }: EventCardProps) {
  const eventDate = parseISO(event.date);
  const today = new Date();
  const isSoon = isAfter(eventDate, today) && isBefore(eventDate, addDays(today, 7));
  const isHappening = event.endDate && isAfter(parseISO(event.endDate), today) && isBefore(eventDate, today);

  return (
    <motion.article
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.4, delay: index * 0.05 }}
      className="group relative"
    >
      <Link href={`/event/${event.id}`}>
        <div className="bg-white rounded-2xl overflow-hidden border-2 border-gray-200 hover:border-[#E11D2E] shadow-[0_4px_20px_rgba(0,0,0,0.04)] hover:shadow-[0_12px_40px_rgba(225,29,46,0.12)] transition-all duration-300">
          {/* Header with date badge */}
          <div className="p-5 relative">
            {/* Top row: Event type & Date */}
            <div className="flex items-start justify-between mb-4">
              <div className="flex items-center gap-2">
                <span className="inline-flex items-center px-2.5 py-1 rounded-full bg-[#E11D2E] text-white text-[10px] font-bold uppercase tracking-wide">
                  {formatEventType(event.eventType)}
                </span>
                <span className="text-xs text-gray-500 font-medium">
                  {event.country}
                </span>
              </div>
              <div className="flex flex-col items-center px-3 py-2 rounded-lg bg-[#FFF5F5] border border-red-100">
                <span className="text-[10px] text-[#E11D2E] uppercase font-semibold">
                  {format(eventDate, 'MMM')}
                </span>
                <span className="text-xl font-bold text-[#E11D2E]">
                  {format(eventDate, 'd')}
                </span>
              </div>
            </div>
            
            {/* Event name */}
            <h3 className="font-heading text-xl font-semibold text-gray-900 mb-2 line-clamp-2 group-hover:text-[#E11D2E] transition-colors">
              {event.name}
            </h3>
            
            {/* Description */}
            <p className="text-gray-600 text-sm mb-4 line-clamp-2">
              {event.description}
            </p>
            
            {/* Meta info */}
            <div className="space-y-2">
              <div className="flex items-center gap-2 text-sm text-gray-500">
                <Calendar className="w-4 h-4 text-[#E11D2E]" />
                <span>
                  {format(eventDate, 'MMM d, yyyy')}
                  {event.endDate && ` - ${format(parseISO(event.endDate), 'MMM d, yyyy')}`}
                </span>
              </div>
              
              <div className="flex items-center gap-2 text-sm text-gray-500">
                <MapPin className="w-4 h-4 text-[#E11D2E]" />
                <span className="truncate">{event.venue}, {event.city}</span>
              </div>
              
              <div className="flex items-center gap-2 text-sm text-gray-500">
                <Users className="w-4 h-4 text-[#E11D2E]" />
                <span>{event.attendeesRange} attendees</span>
              </div>
            </div>

            {/* Status indicator */}
            {isSoon && (
              <div className="absolute top-3 right-20">
                <span className="inline-flex items-center gap-1 px-2 py-1 rounded-full bg-amber-50 text-amber-700 border border-amber-200 text-[10px] font-semibold">
                  <Clock className="w-3 h-3" />
                  This Week
                </span>
              </div>
            )}
            {isHappening && (
              <div className="absolute top-3 right-20">
                <span className="inline-flex items-center gap-1 px-2 py-1 rounded-full bg-green-50 text-green-700 border border-green-200 text-[10px] font-semibold animate-pulse">
                  <span className="w-1.5 h-1.5 rounded-full bg-green-500" />
                  Live Now
                </span>
              </div>
            )}
          </div>
          
          {/* Footer */}
          <div className="px-5 py-4 flex items-center justify-between bg-gray-50 border-t border-gray-100">
            {event.ticketAvailable ? (
              <span className="inline-flex items-center gap-1.5 px-3 py-1.5 rounded-full bg-[#FEE2E2] text-[#E11D2E] text-xs font-semibold uppercase border border-[#FECACA]">
                <Ticket className="w-3 h-3" />
                Tickets Available
              </span>
            ) : (
              <span className="inline-flex items-center gap-1.5 px-3 py-1.5 rounded-full bg-gray-100 text-gray-600 text-xs font-semibold uppercase border border-gray-200">
                <Ticket className="w-3 h-3" />
                Invite Only
              </span>
            )}
            
            <span className="text-sm font-semibold text-gray-900">
              {event.pricing || 'Free Entry'}
            </span>
          </div>
        </div>
      </Link>
    </motion.article>
  );
}

/**
 * Skeleton loader for event cards - Light Theme
 */
export function EventCardSkeleton() {
  return (
    <div className="bg-white rounded-2xl overflow-hidden border-2 border-gray-200 shadow-[0_4px_20px_rgba(0,0,0,0.04)]">
      <div className="p-5">
        <div className="flex items-start justify-between mb-4">
          <div className="h-6 w-24 rounded-full bg-gray-100 animate-pulse" />
          <div className="h-14 w-12 rounded-lg bg-gray-100 animate-pulse" />
        </div>
        <div className="h-6 w-3/4 rounded bg-gray-100 animate-pulse mb-3" />
        <div className="h-4 w-full rounded bg-gray-100 animate-pulse mb-2" />
        <div className="h-4 w-2/3 rounded bg-gray-100 animate-pulse mb-4" />
        <div className="space-y-2">
          <div className="h-4 w-1/2 rounded bg-gray-100 animate-pulse" />
          <div className="h-4 w-2/3 rounded bg-gray-100 animate-pulse" />
          <div className="h-4 w-1/3 rounded bg-gray-100 animate-pulse" />
        </div>
      </div>
      <div className="px-5 py-4 flex items-center justify-between bg-gray-50 border-t border-gray-100">
        <div className="h-7 w-32 rounded-full bg-gray-100 animate-pulse" />
        <div className="h-5 w-20 rounded bg-gray-100 animate-pulse" />
      </div>
    </div>
  );
}
